#include "SSH.h"

namespace Upp {

#define LLOG(x)	 	RLOG(x)

void Channel::StartInit()
{
	AddJob() << [=] {
	 	channel = libssh2_channel_open_session(ssh->GetSession());
	 	if(channel) {
	 		LLOG("++ Channel: Session successfully opened.");
	 		return false;
	 	}
		if(!WouldBlock()) Error();
		return true;
	};
}

void Channel::StartStop()
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_free(channel);
		if(rc == 0) {
			LLOG("++ Channel: Handle succesfully freed.");
			channel = NULL;
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
}

Channel& Channel::StartRequest(const String& request, const String& params)
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_process_startup(
			channel,
			request,
			request.GetLength(),
			params,
			params.GetLength()
		);
		if(rc == 0) {
			LLOG("++ Channel: " << request << " request is successful.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartExec(const String& cmdline)
{
	return StartRequest("exec", cmdline);
}

Channel& Channel::StartShell()
{
	return StartRequest("shell", Null);
}

Channel& Channel::StartSubsystem(const String& subsystem)
{
	return StartRequest("subsystem", subsystem);
}

Channel& Channel::StartTerminal(const String& term, int width, int height)
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_request_pty(channel, ~term);
		if(rc == 0) {
			LLOG("++ Channel: Terminal opened.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};

	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_request_pty_size(channel, width, height);
		if(rc == 0) {
			LLOG("++ Channel: Terminal size adjusted. (W:" << width << ", H:" << height << ")");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartSetEnv(const String& variable, const String& value)
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_setenv(channel, variable, value);
		if(rc == 0) {
			LLOG("++ Channel: Environment variable " << variable << " set to " << value);
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartRead(Stream& in, int64 size, Gate<int64, int64> progress)
{
	Stream *p = &in;
	packet_length = 0;
	AddJob() << [=] { return DataRead(0, *p, size, progress); };
	return *this;
}

Channel& Channel::StartWrite(Stream& out, int64 size, Gate<int64, int64> progress)
{
	Stream *p = &out;
	packet_length = 0;
	AddJob() << [=] { return DataWrite(0, *p, size, progress); };
	return *this;
}

bool Channel::DataRead(int id, Stream& out, int64 size, Gate<int64, int64> progress)
{
	ASSERT(channel);
	auto remaining = size - out.GetSize();
	chunk_size = chunk_size > remaining
		? remaining
		: chunk_size;
	Buffer<char> buffer(chunk_size);
	int rc = libssh2_channel_read_ex(channel, id, buffer, chunk_size);
	if(rc < 0) {
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	}
	if(rc == 0)
		return false;
	out.Put(buffer, rc);
	if(progress(size, out.GetSize()))
		Error(-1, t_("Read aborted."));
	if(size == out.GetSize()) {
		LLOG(Format("++ Channel: %ld of %ld bytes successfully read.",
				out.GetSize(),
				size));
		return false;
	}
	return true;
}

bool Channel::DataWrite(int id, Stream& in, int64 size, Gate<int64, int64> progress)
{
	ASSERT(channel);
	auto rc = libssh2_channel_write_ex(channel, id ,in.Get(chunk_size), chunk_size);
	if(rc >= 0) {
		packet_length += rc;
		in.Seek(packet_length);
		if(progress(in.GetSize(), in.GetPos()))
			Error(-1, t_("Write aborted."));
		if(in.IsEof()) {
			LLOG(Format("++ Channel: %ld of %ld bytes successfully written.",
					in.GetPos(),
					in.GetSize()));
			return false;
		}
	}
	if(rc != LIBSSH2_ERROR_EAGAIN) Error();
	return true;
}

Channel& Channel::StartClose()
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_close(channel);
		if(rc == 0) {
			LLOG("++ Channel: Closing...");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartCloseWait()
{
	StartClose();

	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_wait_closed(channel);
		if(rc == 0) {
			LLOG("++ Channel: Closed successfully.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartSendEof()
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_send_eof(channel);
		if(rc == 0) {
			LLOG("++ Channel: EOF message sent to server.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartRecvEof()
{
	AddJob() << [=] {
		ASSERT(channel);
		int rc = libssh2_channel_wait_eof(channel);
		if(rc == 0) {
			LLOG("++ Channel: EOF message acknowledged by the server.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Channel& Channel::StartSendRecvEof()
{
	StartSendEof();
	StartRecvEof();
	return *this;

}

Channel& Channel::StartReadStdErr(Stream& err)
{
	Stream *p = &err;
	AddJob() << [=] { return DataRead(SSH_EXTENDED_DATA_STDERR, *p, chunk_size); };
	return *this;
}

Channel& Channel::StartWriteStdErr(Stream& err)
{
	Stream *p = &err;
	AddJob() << [=] { return DataWrite(SSH_EXTENDED_DATA_STDERR, *p, chunk_size); };
	return *this;
}
Channel& Channel::StartGetExitCode()
{
	AddJob() << [=] { ASSERT(channel); GetExitCode(); return false; };
	return *this;
}

Channel& Channel::StartGetExitSignal()
{
	AddJob() << [=] { ASSERT(channel); GetExitSignal(); return false; };
	return *this;
}

int Channel::GetExitCode()
{
	if(channel) {
		int rc = libssh2_channel_get_exit_status(channel);
		if(rc == 0)
			LLOG("++ Channel: Exit code: " << rc);
		code = rc;
	}
	return code;
}

String Channel::GetExitSignal()
{
	signal.SetLength(64);
 	char* sig = signal.Begin();
	if(channel)
   	 libssh2_channel_get_exit_signal(channel, &sig, NULL, NULL, NULL, NULL, NULL);
	return signal;
}

void Channel::CleanUp()
{
	if(channel) StartStop();
	if(!IsCleanup()) Execute();
	else LLOG("** Channel: Performing clean up...");
}

Channel::Channel()
{
	type = Type::CHANNEL;
	channel = NULL;
	code = Null;
}
}